/*! \file    
**********************************************************************************	
* Title:						Discretix SST API Source file						 					
*																			
* Filename:						sst_api_data_operations.c															
*																			
* Project, Target, subsystem:	SST Host IF, APIs
* 
* Created:						01.07.2007															
*
* Modified:						07.06.2007										
*
* \Author						Ira Boguslavsky														
*																			
* \Remarks						
*           Copyright (C) 2006 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/

/*----------- External include files ----------------------------------------*/
#include "DX_VOS_Mem.h"
#include "SEPDriver.h"
#include "sst_errors.h"
#include "sst_sep_errors.h"
#include "sst_types.h"
#include "error.h"
#include "sst_host_util.h"
#include "sst_host_def.h"
/*----------- Local include files -------------------------------------------*/
#include "sst_test_flags.h"
#include "sst_host_op_code.h"

/*----------- Local include files -------------------------------------------*/
/*----------- Local Variable declarations -----------------------------------*/
/*----------- Local function declarations -----------------------------------*/
/*----------- Local constant definitions  -----------------------------------*/
  
/*---------------------------------------------------------------------------*/
/*               API FUNCTIONS                                               */
/*---------------------------------------------------------------------------*/

/*Basic DB services*/

/*SST_DataInsert*/
DxError_t SST_DataInsert(SSTTxnId_t        aTransactionId,
                         SSTSessionId_t    aSessionId, 
                         SSTHandle_t       aInitialCtrlAuth,       
                         DxByte_t         *aDataIn_ptr,
                         DxUint32_t        aDataInsizeInBytes,
                         DxBool_t          aEncrypt,
                         SSTDataType_t     aDataType,                
                         SSTHandle_t      *aDataHandle_ptr,
                         DxByte_t         *aWorkspace_ptr,
                         DxUint32_t        aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_11_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_4_WORDS_PARAMS];
    DxError_t               errorRC;
	DxUint32_t				dataInPhySM;	
    DxUint32_t				dataInVirSM;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*allocate memory in SM*/  /* Ira */
	errorRC = SEPDriver_AllocateDataPoolMemory( aDataIn_ptr, 
                                                aDataInsizeInBytes, 
                                               &dataInPhySM,
                                               &dataInVirSM);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_data_insert);
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(dataInVirSM,
                                              aDataIn_ptr, 
                                              aDataInsizeInBytes);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_data_insert);
	
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_DATA_INSERT;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)dataInPhySM;
    paramInBuffer_ptr[6] =  (DxUint32_t)aDataInsizeInBytes;
    paramInBuffer_ptr[7] =  (DxUint32_t)aEncrypt;
    paramInBuffer_ptr[8] =  (DxUint32_t)aDataType;
    paramInBuffer_ptr[9] =  (DxUint32_t)aDataHandle_ptr->objDigest;
    paramInBuffer_ptr[10] = (DxUint32_t)aDataHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_11_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        DX_4_WORDS_PARAMS * sizeof(DxUint32_t));
    if (DX_OK != errorRC)
        goto exit_data_insert;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_DATA_INSERT)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_data_insert;
    }
    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_data_insert);

    /* Fill out output parameters */
    aDataHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aDataHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_data_insert: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
    
    return errorRC;
}


/*SST_DataReplace*/
DxError_t SST_DataReplace(  SSTTxnId_t      aTransactionId,
                            SSTSessionId_t  aSessionId,                           
                            SSTHandle_t     aDataHandle,   
                            DxByte_t       *aNewData_ptr,
                            DxUint32_t      aNewDataSizeInBytes,
                            DxBool_t        aEncrypt,
                            DxByte_t       *aWorkspace_ptr,
                            DxUint32_t      aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_8_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_2_WORDS_PARAMS];
    DxError_t               errorRC;
	DxUint32_t				dataInPhySM;	
    DxUint32_t				dataInVirSM;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*allocate memory in SM*/ 
	errorRC = SEPDriver_AllocateDataPoolMemory( aNewData_ptr, 
                                                aNewDataSizeInBytes, 
                                               &dataInPhySM,
                                               &dataInVirSM);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_data_replace);
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(dataInVirSM,
                                              aNewData_ptr, 
                                              aNewDataSizeInBytes);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_data_replace);
	
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_DATA_REPLACE;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aDataHandle.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aDataHandle.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)dataInPhySM;
    paramInBuffer_ptr[6] =  (DxUint32_t)aNewDataSizeInBytes;
    paramInBuffer_ptr[7] =  (DxUint32_t)aEncrypt;
    
    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_8_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        DX_2_WORDS_PARAMS * sizeof(DxUint32_t));
    if (DX_OK != errorRC)
        goto exit_data_replace;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_DATA_REPLACE)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_data_replace;
    }
    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_data_replace);
    
exit_data_replace: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:   
    return errorRC;
}

/*SST_DataReadOnlySet*/
DxError_t SST_DataReadOnlySet(SSTTxnId_t      aTransactionId,
                              SSTSessionId_t  aSessionId, 
                              SSTHandle_t     aObjHandle,
                              DxByte_t       *aWorkspace_ptr,
                              DxUint32_t      aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_5_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_2_WORDS_PARAMS];
    DxError_t               errorRC;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_DATA_READ_ONLY_SET;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] = (DxUint32_t)aObjHandle.objDigest;
    paramInBuffer_ptr[4] = (DxUint32_t)aObjHandle.objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_5_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        DX_2_WORDS_PARAMS * sizeof(DxUint32_t));
    if (DX_OK != errorRC)
        goto exit_read_only_set;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_DATA_READ_ONLY_SET)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_read_only_set;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_read_only_set);

exit_read_only_set: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
    
    return errorRC;
}

/*SST_DataRead*/
DxError_t SST_DataRead(SSTSessionId_t  aSessionId,
                       SSTHandle_t     aDataHandle,
                       DxUint32_t      aDataOffset,
                       DxByte_t       *aDataOut_ptr,
                       DxUint32_t     *aDataSizeInBytes_ptr,
                       DxByte_t       *aWorkspace_ptr,
                       DxUint32_t      aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_7_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_3_WORDS_PARAMS];
    DxError_t               errorRC;
    DxError_t               sepErrorRC;
    DxUint32_t				dataOutSMPhyAddress;
    DxUint32_t				dataOutSMVirtAddress;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
    
    if(DX_IS_NULL_2PARAMS(aDataOut_ptr,aDataSizeInBytes_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*allocate in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory(   aDataOut_ptr, 
                                                 *aDataSizeInBytes_ptr, /* Ira */
                                                 &dataOutSMPhyAddress,
                                                 &dataOutSMVirtAddress);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_read);
	
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_DATA_READ;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aDataHandle.objDigest;
    paramInBuffer_ptr[3] = (DxUint32_t)aDataHandle.objId;
    paramInBuffer_ptr[4] = (DxUint32_t)aDataOffset;
    paramInBuffer_ptr[5] = (DxUint32_t)dataOutSMPhyAddress;
    paramInBuffer_ptr[6] = (DxUint32_t)(*aDataSizeInBytes_ptr);

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_7_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        DX_3_WORDS_PARAMS * sizeof(DxUint32_t));
    if (DX_OK != errorRC)
        goto exit_read;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_DATA_READ)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_read;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_read);

    /* Fill out output parameters */
    *aDataSizeInBytes_ptr = paramOutBuffer_ptr[2];

	/*copy from SM*/
	sepErrorRC = SEPDriver_ReadFromDataPoolMemory(dataOutSMVirtAddress, 
									           aDataOut_ptr, 
									           *aDataSizeInBytes_ptr);
	SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_read);

exit_read:   
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
  
    return errorRC;
}

/*SST_DataModify*/
DxError_t SST_DataModify(SSTTxnId_t      aTransactionId,
                         SSTSessionId_t  aSessionId,                           
                         SSTHandle_t     aDataHandle,   
                         DxByte_t       *aDataIn_ptr,
                         DxUint32_t      aDataSizeInBytes,
                         DxUint32_t      aDataOffset,
                         DxByte_t       *aWorkspace_ptr,
                         DxUint32_t      aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_8_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_2_WORDS_PARAMS];
    DxError_t               errorRC;
    DxUint32_t				dataInSMPhy;
    DxUint32_t 				dataInSMVirt;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*allocate memory in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory( aDataIn_ptr, 
                                                aDataSizeInBytes, 
                                                &dataInSMPhy,
                                                &dataInSMVirt);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_data_modify);
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(dataInSMVirt,aDataIn_ptr, aDataSizeInBytes);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_data_modify);
	
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_DATA_MODIFY;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] = (DxUint32_t)aDataHandle.objDigest; 
    paramInBuffer_ptr[4] = (DxUint32_t)aDataHandle.objId; 
    paramInBuffer_ptr[5] = (DxUint32_t)dataInSMPhy;
    paramInBuffer_ptr[6] = (DxUint32_t)aDataSizeInBytes;
    paramInBuffer_ptr[7] = (DxUint32_t)aDataOffset;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_8_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        DX_2_WORDS_PARAMS * sizeof(DxUint32_t));  
    if (DX_OK != errorRC)
        goto exit_data_modify;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_DATA_MODIFY)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_data_modify;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_data_modify);

exit_data_modify: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();
    
exit_function:
    
    return errorRC;
}


/*SST_DataSizeGet*/
DxError_t SST_DataSizeGet(SSTSessionId_t  aSessionId,  
                          SSTHandle_t     aDataHandle,
                          DxUint32_t     *aDatasizeInBytes_ptr,
                          DxByte_t       *aWorkspace_ptr,
                          DxUint32_t      aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_4_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_3_WORDS_PARAMS];
    DxError_t               errorRC;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    if(DX_IS_NULL_PARAM(aDatasizeInBytes_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_DATA_SIZE_GET;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aDataHandle.objDigest;
    paramInBuffer_ptr[3] = (DxUint32_t)aDataHandle.objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_4_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        DX_3_WORDS_PARAMS * sizeof(DxUint32_t));  
    if (DX_OK != errorRC)
        goto exit_data_size_get;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_DATA_SIZE_GET)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_data_size_get;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_data_size_get);

    /* Fill out output parameter */
    *aDatasizeInBytes_ptr = paramOutBuffer_ptr[2];

exit_data_size_get:
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
     
    return errorRC;
}

/*SST_DataDelete*/
DxError_t SST_DataDelete(SSTTxnId_t      aTransactionId,
                         SSTSessionId_t  aSessionId,  
                         SSTHandle_t     aDataHandle,
                         DxByte_t       *aWorkspace_ptr,
                         DxUint32_t      aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_5_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_2_WORDS_PARAMS];
    DxError_t               errorRC;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_DATA_DELETE;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] = (DxUint32_t)aDataHandle.objDigest;
    paramInBuffer_ptr[4] = (DxUint32_t)aDataHandle.objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_5_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        DX_2_WORDS_PARAMS * sizeof(DxUint32_t));  
    if (DX_OK != errorRC)
        goto exit_data_delete;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_DATA_DELETE)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_data_delete;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_data_delete);

exit_data_delete: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
    
    return errorRC;
}



/*SST_TransactionStart*/
DxError_t SST_TransactionStart(SSTTxnId_t *aTransactionId_ptr)
{
    DxUint32_t              paramInBuffer_ptr [DX_1_WORD_PARAM];
    DxUint32_t              paramOutBuffer_ptr[DX_3_WORDS_PARAMS];
    DxError_t               errorRC;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_TRANSACTION_START;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
                                        /* Input buffer length */
                                        DX_1_WORD_PARAM * sizeof(DxUint32_t), 
                                        paramOutBuffer_ptr,
                                        /* Output buffer maximal length */
                                        DX_3_WORDS_PARAMS * sizeof(DxUint32_t));  
    if (DX_OK != errorRC)
        goto exit_transaction_start;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_TRANSACTION_START)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_transaction_start;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_transaction_start);

    /* Output parameter */
    *aTransactionId_ptr = (SSTTxnId_t)paramOutBuffer_ptr[2];

exit_transaction_start: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();
    
exit_function:
    
    return errorRC;
}

/*SST_TransactionEnd*/
DxError_t SST_TransactionEnd(SSTTxnId_t aTransactionId,
                             DxBool_t   aIsReflashSensitive,
                             DxByte_t   *aWorkspace_ptr,
                             DxUint32_t aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_3_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_2_WORDS_PARAMS];
    DxError_t               errorRC;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_TRANSACTION_END;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aIsReflashSensitive;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_3_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        DX_2_WORDS_PARAMS * sizeof(DxUint32_t));  

    if (DX_OK != errorRC)
        goto exit_transaction_end;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_TRANSACTION_END)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_transaction_end;
    }

    /* RC */    
    errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_transaction_end);

exit_transaction_end: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
    
    return errorRC;
}

/*SST_TransactionRollback*/
DxError_t SST_TransactionRollback(SSTTxnId_t  aTransactionId,
                                  DxByte_t   *aWorkspace_ptr,
                                  DxUint32_t  aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr [DX_2_WORDS_PARAMS];
    DxUint32_t              paramOutBuffer_ptr[DX_2_WORDS_PARAMS];
    DxError_t               errorRC;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
    
	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_TRANSACTION_ROLLBACK;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aTransactionId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        DX_2_WORDS_PARAMS * sizeof(DxUint32_t), 
        paramOutBuffer_ptr,
        /* Output buffer length */
        DX_2_WORDS_PARAMS * sizeof(DxUint32_t));  
    if (DX_OK != errorRC)
        goto exit_transaction_rollback;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_TRANSACTION_ROLLBACK)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_transaction_rollback;
    }

    /* RC */    
    errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_transaction_rollback);

exit_transaction_rollback:  
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
   
    return errorRC;
}



